<?php

namespace App\Http\Controllers;

use App\Models\JobQuestion;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class JobQuestionController extends Controller
{
    public function index(): JsonResponse
    {
        $questions = JobQuestion::with('job')->get();
        return response()->json([
            'status' => 'success',
            'data' => $questions,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $question = JobQuestion::with('job')->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $question,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Job_id' => 'required|exists:jobs,Job_id',
            'Question_text' => 'required|string',
            'Question_type' => 'sometimes|in:text,multiple_choice,single_choice,rating_scale,boolean',
            'Options' => 'nullable|array',
            'Options.*' => 'string',
            'Order' => 'sometimes|integer|min:0',
            'Is_required' => 'sometimes|boolean',
            'Max_length' => 'nullable|integer|min:1',
            'Min_value' => 'nullable|integer',
            'Max_value' => 'nullable|integer',
            'Validation_rules' => 'nullable|array',
            'Validation_rules.*' => 'string',
            'Weight' => 'sometimes|numeric|between:0,99.99',
        ], [
            'Question_type.in' => 'The question type must be one of: text, multiple_choice, single_choice, rating_scale, boolean.',
            'Options.array' => 'Options must be an array of strings.',
            'Order.min' => 'Order must be a non-negative integer.',
            'Max_length.min' => 'Max length must be at least 1.',
            'Weight.between' => 'Weight must be between 0 and 99.99.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'Job_id',
            'Question_text',
            'Question_type',
            'Options',
            'Order',
            'Is_required',
            'Max_length',
            'Min_value',
            'Max_value',
            'Validation_rules',
            'Weight',
        ]);

        // Set default values if not provided
        if (!$request->has('Question_type')) {
            $data['Question_type'] = 'text';
        }
        if (!$request->has('Order')) {
            $data['Order'] = 0;
        }
        if (!$request->has('Is_required')) {
            $data['Is_required'] = true;
        }
        if (!$request->has('Weight')) {
            $data['Weight'] = 1.00;
        }

        $question = JobQuestion::create($data);

        return response()->json([
            'status' => 'success',
            'data' => $question->load('job'),
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $question = JobQuestion::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Job_id' => 'sometimes|exists:jobs,Job_id',
            'Question_text' => 'sometimes|string',
            'Question_type' => 'sometimes|in:text,multiple_choice,single_choice,rating_scale,boolean',
            'Options' => 'nullable|array',
            'Options.*' => 'string',
            'Order' => 'sometimes|integer|min:0',
            'Is_required' => 'sometimes|boolean',
            'Max_length' => 'nullable|integer|min:1',
            'Min_value' => 'nullable|integer',
            'Max_value' => 'nullable|integer',
            'Validation_rules' => 'nullable|array',
            'Validation_rules.*' => 'string',
            'Weight' => 'sometimes|numeric|between:0,99.99',
        ], [
            'Question_type.in' => 'The question type must be one of: text, multiple_choice, single_choice, rating_scale, boolean.',
            'Options.array' => 'Options must be an array of strings.',
            'Order.min' => 'Order must be a non-negative integer.',
            'Max_length.min' => 'Max length must be at least 1.',
            'Weight.between' => 'Weight must be between 0 and 99.99.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $question->update($request->only([
            'Job_id',
            'Question_text',
            'Question_type',
            'Options',
            'Order',
            'Is_required',
            'Max_length',
            'Min_value',
            'Max_value',
            'Validation_rules',
            'Weight',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $question->load('job'),
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $question = JobQuestion::findOrFail($id);
        $question->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Job question deleted successfully',
        ], 200);
    }
}
